// #region UTILS
const Utils = {
    /**
     * Formatea un número con separadores de miles y dos decimales.
     * @param {number|string} x - El número a formatear.
     * @returns {string} El número formateado.
     */
    Separador: function(x) { //SEPARADOR CON DECIMAL
        return parseFloat(x).toFixed(0).replace(/\B(?=(\d{3})+(?!\d))/g, ".");
    },
    /**
     * Permite solo la entrada de números y un punto decimal en un campo de texto.
     * @param {Event} event - El objeto de evento del teclado.
     * @returns {boolean} True si la tecla presionada es un número o un punto, false en caso contrario.
     */
    SoloNumeros: function(event) {
        const charCode = (event.which) ? event.which : event.keyCode;

        // Permite números (0-9)
        // Permite la tecla de retroceso (Backspace, charCode 8) para borrar
        // Permite la tecla de suprimir (Delete, charCode 46, pero aquí también es el punto)
        // Permite la tecla de tabulación (Tab, charCode 9) para navegar
        if (charCode >= 48 && charCode <= 57) { // Números 0-9
            return true;
        }

        // Permite el punto decimal si no hay uno ya en el campo
        if (charCode === 46) { // Tecla de punto (o suprimir en algunos casos)
            // Asegura que no se introduzcan múltiples puntos decimales
            if (event.target.value.indexOf('.') === -1) {
                return true;
            }
        }
        
        // Permite las teclas de navegación (flechas, inicio, fin)
        // charCode 37: Left arrow
        // charCode 39: Right arrow
        // charCode 8: Backspace
        // charCode 9: Tab
        // charCode 46: Delete (y el punto, ya manejado arriba)
        if (charCode === 8 || charCode === 9 || charCode === 37 || charCode === 39) {
            return true;
        }

        // Si no es un número ni un punto ni una tecla de control permitida
        return false;
    },
    /**
     * Limpia los campos de texto de entrada de productos después de agregar al carrito.
     */
    LimpiarTexto: function() {
        $("#search_producto_combo").val("");
        $("#idproducto").val("");
        $("#codproducto").val("");
        $("#producto").val("");
        $("#opcionvendido").val("");
        $("#codmarca").val("");
        $("#marca").val("");
        $("#codmodelo").val("");
        $("#modelo").val("");
        $("#codpresentacion").val("");
        $("#existenciap").val("0.00");
        $("#presentacion").val("");
        $("#preciocomprap").val("");
        $("#precioventap").val("");
        $("#cantidad").val("1");
    },
    /**
     * Resetea los estilos de los campos de entrada de validación.
     */
    resetInputStyles: function() {
        $("#cantidad, #search_producto_combo").css('border-color', '');
    },
    /**
     * Muestra un mensaje de alerta.
     * @param {string} message - El mensaje a mostrar.
     * @param {string} type - Tipo de alerta ('success', 'error', 'warning', 'info').
     * @param {string|null} inputId - ID del input para enfocar y resaltar (opcional).
     */
    showAlert: function(message, type = "error", inputId = null) {
        swal("Oops", message, type);
        if (inputId) {
            $(inputId).focus();
            $(inputId).css('border-color', '#ff7676');
        }
    }
};
// #endregion UTILS

// #region CART CALCULATOR
const _cartCalculator = {
    /**
     * Calcula los totales del carrito de ventas basándose en los ítems y los parámetros globales de descuento/impuestos.
     * @param {Array<Object>} items - Array de objetos que representan los productos en el carrito.
     * @returns {Object} Un objeto con todos los totales calculados.
     */
    calculateTotals: function(items) {

        // Inicializa un objeto 'totals' con todos los valores a 0
        let totals = {
            OperacionItems:    0,
            TotalDescuentoIndividual: 0,
            TotalCompra:      0,
            TotalVenta:       0
        };

        // Si el array de ítems está vacío, simplemente retornamos el objeto 'totals' con todos sus valores en cero.
        if (!items || items.length === 0) {
            return totals;
        }

        let currentOperacionItems           = 0;
        let currentTotalDescuentoIndividual = 0;
        let currentTotalCompra              = 0;
        let currentTotalVenta               = 0;

        items.forEach(item => {
            if (parseFloat(item.cantidad) !== 0) {
                const cantsincero       = parseFloat(item.cantidad);
                currentOperacionItems  += cantsincero;

                const TotalCompra       = parseFloat(item.precio) * parseFloat(item.cantidad);
                //const currentSumaCompra = parseFloat(currentSumaCompra) + parseFloat(TotalCompra);
                const OperacionCompra   = parseFloat(item.precio) * cantsincero;
                currentTotalCompra     += OperacionCompra;

                const TotalVenta        = parseFloat(item.precio2) * parseFloat(item.cantidad);
                //const currentSumaVenta  = parseFloat(currentSumaVenta) + parseFloat(TotalVenta);
                const OperacionVenta    = parseFloat(item.precio2) * cantsincero;
                currentTotalVenta       += OperacionVenta;

                //const descsiniva        = PrecioVenta * parseFloat(item.descproducto) / 100;
                //currentTotalDescuentoIndividual += ValorTotal * parseFloat(item.descproducto) / 100;
            }
        });

        // Asignar los valores calculados al objeto 'totals' que se inicializó al principio
        totals.OperacionItems           = currentOperacionItems;
        totals.TotalDescuentoIndividual = Math.round(currentTotalDescuentoIndividual);
        totals.TotalCompra              = Math.round(currentTotalCompra);
        totals.TotalVenta               = Math.round(currentTotalVenta);

        return totals;
    }
};
// #endregion CART CALCULATOR

// #region UI UPDATER
const urlParams       = new URLSearchParams(window.location.search);
const currentCodCombo = urlParams.get('codcombo');
//const currentProceso  = urlParams.get('proceso');
const isEditingMode   = currentCodCombo;

const _uiUpdater = {
    /**
     * Renderiza la tabla del carrito con los items y actualiza los resúmenes.
     * @param {Array<Object>} items - Array de objetos que representan los productos en el carrito.
     * @param {Object} totals - Objeto con los totales calculados por _cartCalculator.
     */
    updateCartUI: function(items, totals) {

        //console.log("-----------------------------------------");
        //console.log("DEBUG UI UPDATER: Totales recibidos para actualizar UI:", totals);
        //console.log("DEBUG UI UPDATER: Items recibidos para actualizar UI:", items);

        $("#carrito tbody").html("");
        if (items.length === 0) {
            this.resetSalesUI();
            return; // Importante para salir de la funció
        } else {
            items.forEach(item => {
                const cantsincero = parseFloat(item.cantidad);
                if (cantsincero !== 0) {

                    const PrecioCompra    = parseFloat(item.precio);
                    const ValorCompra     = PrecioCompra * cantsincero;
                    const DescuentoCompra = ValorCompra * parseFloat(item.descproducto) / 100;
                    const OperacCompra    = PrecioCompra;
                    const OperacionCompra = OperacCompra * cantsincero;

                    const PrecioVenta     = parseFloat(item.precio2);
                    const ValorVenta      = PrecioVenta * cantsincero;
                    const DescuentoVenta  = ValorVenta * parseFloat(item.descproducto) / 100;
                    const OperacVenta     = PrecioVenta;
                    const OperacionVenta  = OperacVenta * cantsincero;

                    const nuevaFila = `
                        <tr class='warning-element' style='border-left: 2px solid #ff5050 !important; background: #fce3e3;' align='center'>
                            <td>
                                <button class="btn btn-info btn-sm" style="cursor:pointer;border-radius:5px 0px 0px 5px;"
                                onclick="addItem('${item.id}', '${item.txtCodigo}', '-1', '${item.producto}', '${item.opcionvendido}', '${item.codmarca}', '${item.marca}', '${item.codmodelo}', '${item.modelo}', '${item.codpresentacion}', '${item.presentacion}', '${item.precio}', '${item.precio2}', '-' )"
                                type='button'><span class='fa fa-minus'></span></button><input type='text' class='bold item-qty-input'
                                data-item-id='${item.id}'
                                data-item-codigo='${item.txtCodigo}'
                                data-item-producto='${item.producto.replace(/'/g, "&apos;")}'
                                data-item-opcionvendido='${item.opcionvendido}'
                                data-item-codmarca='${item.codmarca}'
                                data-item-marca='${item.marca.replace(/'/g, "&apos;")}'
                                data-item-codmodelo='${item.codmodelo}'
                                data-item-modelo='${item.modelo.replace(/'/g, "&apos;")}'
                                data-item-codpresentacion='${item.codpresentacion}'
                                data-item-presentacion='${item.presentacion.replace(/'/g, "&apos;")}'
                                data-item-precio='${item.precio}'
                                data-item-precio2='${item.precio2}'
                                data-item-descproducto='${item.descproducto}'
                                value='${item.cantidad}'
                                style='width:50px;height:29px;'
                                onchange='handleQuantityChange(this)'
                                onkeypress="return Utils.SoloNumeros(event)"><button class="btn btn-info btn-sm" style="cursor:pointer;border-radius:0px 5px 5px 0px;"
                                onclick="addItem('${item.id}', '${item.txtCodigo}', '+1', '${item.producto}', '${item.opcionvendido}', '${item.codmarca}', '${item.marca}', '${item.codmodelo}', '${item.modelo}', '${item.codpresentacion}', '${item.presentacion}', '${item.precio}', '${item.precio2}', '${item.descproducto}', '+' )"
                                type='button'><span class='fa fa-plus'></span></button>
                            </td>
                            <td class='text-left'><h5><strong>${item.producto}</strong></h5></td>
                            <td><strong>${Utils.Separador(PrecioCompra)}</strong></td>
                            <td><strong>${Utils.Separador(PrecioVenta)}</strong></td>
                            <td><strong>${Utils.Separador(OperacionCompra)}</strong></td>
                            <td><strong>${Utils.Separador(OperacionVenta)}</strong></td>
                            <td>
                                <span class="text-danger" title="Eliminar Detalle" style="cursor:pointer;color:#fff;"
                                onclick="addItem('${item.id}', '${item.txtCodigo}', '0', '${item.producto}', '${item.opcionvendido}', '${item.codmarca}', '${item.marca}', '${item.codmodelo}', '${item.modelo}', '${item.codpresentacion}', '${item.presentacion}', '${item.precio}', '${item.precio2}', '${item.descproducto}', '=' )"
                                title="Eliminar">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-trash-2 icon"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path><line x1="10" y1="11" x2="10" y2="17"></line><line x1="14" y1="11" x2="14" y2="17"></line></svg>
                                </span>
                            </td>
                        </tr>
                    `;
                    $(nuevaFila).appendTo("#carrito tbody");
                }
            });
            $("#lblcosto").text(Utils.Separador(totals.TotalCompra));
            $("#lblventa").text(Utils.Separador(totals.TotalVenta));
        }
        Utils.LimpiarTexto();
    },

    /**
     * Limpia completamente la interfaz de usuario de ventas.
     */
    resetSalesUI: function() {
        $("#carrito tbody").html("<tr class='warning-element' style='border-left: 2px solid #ff5050 !important; background: #fce3e3;'><td class='text-center' colspan=7><h4>NO HAY DETALLES AGREGADOS</h4></td></tr>");
        $("#lblcosto").text("0.00");
        $("#lblventa").text("0.00");
        Utils.LimpiarTexto();
    }
};
// #endregion UI UPDATER

// #region CART MANAGER
const _cartManager = {
    /**
     * Envia una solicitud AJAX para manipular el carrito en el servidor.
     * @param {Object} cartItem - Objeto con los datos del producto/servicio y la operación.
     */
    updateCart: function(cartItem) {
        const DatosJson = JSON.stringify(cartItem);

        $.ajax({ // Cambiamos de $.post a $.ajax para un control más preciso
            url: "carritoproducto.php",
            method: "POST", // Especificamos el método
            data: { MiCarrito: DatosJson },
            dataType: "json", // ¡Esto es CRUCIAL para asegurar que 'data' sea un JSON parseado!
            success: function(data) {
                //console.log("CART MANAGER: Datos recibidos de carritoproducto.php (SUCCESS AJAX):", data);
                if (!data || data.length === 0) { // Si la respuesta es un array vacío
                    //console.log("DEBUG: _cartManager detectó carrito vacío desde PHP. Forzando UI a cero.");
                    _uiUpdater.resetSalesUI(); // Llama a la función que pone los campos de totales a "0.00"
                    return; // Terminamos aquí
                }
                // Si 'data' NO está vacío, procedemos con el cálculo y la actualización normal
                const totals = _cartCalculator.calculateTotals(data);
                //console.log("CART MANAGER: Totales calculados (SUCCESS AJAX):", totals);
                _uiUpdater.updateCartUI(data, totals);
                //console.log("CART MANAGER: _uiUpdater.updateCartUI debería haber actualizado los totales.");
            },
            error: function(jqXHR, textStatus, errorThrown) {
                //console.error("Error al actualizar el carrito:", textStatus, errorThrown);
                Utils.showAlert("Ha Ocurrido un Error al comunicarse con el servidor de carrito!", "error");
            }
        });
    },

    /**
     * Vacia el carrito de ventas (nuevo) en el servidor y actualiza la UI.
     */
    clearCart: function() {
        const Carrito = {
            Id: "vaciar",
            Codigo: "vaciar",
            Producto: "vaciar",
            OpcionVendido: "vaciar",
            Codmedida: "vaciar",
            Medida: "vaciar",
            Precio: "0",
            Precio2: "0",
            Descproducto: "0",
            Cantidad: "0"
        };
        const DatosJson = JSON.stringify(Carrito);
        $.ajax({ // Cambiamos a $.ajax aquí también
            url: "carritoproducto.php",
            method: "POST",
            data: { MiCarrito: DatosJson },
            dataType: "json",
            success: function(data) {
                //console.log("CART MANAGER: Respuesta de vaciar carrito (SUCCESS AJAX):", data);
                if (!data || data.length === 0) {
                    //$("#savetraspaso")[0].reset();
                    _uiUpdater.resetSalesUI(); // Reinicia la UI
                } else {
                    const totals = _cartCalculator.calculateTotals(data);
                    _uiUpdater.updateCartUI(data, totals);
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                //console.error("Error al vaciar el carrito:", textStatus, errorThrown);
                Utils.showAlert("Ha Ocurrido un Error al comunicarse con el servidor para vaciar el carrito!", "error");
            }
        });
    }
};
// #endregion CART MANAGER

// #region CORE FUNCTIONS (Manteniendo la API original)
/**
* Agrega o modifica un item en el carrito. Esta es la función principal que se llama desde los botones y la tabla.
*/
function addItem(idproducto, codproducto, cantidad, producto, opcionvendido, codmarca, marca, codmodelo, modelo, codpresentacion, presentacion, preciocompra, precioventa, descproducto, opCantidad) { //

    const Carrito = {
        Id: idproducto,
        Codigo: codproducto,
        Producto: producto,
        OpcionVendido: opcionvendido,
        Codmarca: codmarca,
        Marca: marca,
        Codmodelo: codmodelo,
        Modelo: modelo,
        Codpresentacion: codpresentacion,
        Presentacion: presentacion,
        Precio: preciocompra,
        Precio2: precioventa,
        Descproducto: descproducto,
        Cantidad: cantidad,
        opCantidad: opCantidad
    };
    _cartManager.updateCart(Carrito);
}

/**
* Realiza la acción de añadir un producto con cantidad 1 al carrito.
*/
function DoAction(idproducto, codproducto, producto, opcionvendido, codmarca, marca, codmodelo, modelo, codpresentacion, presentacion, preciocompra, precioventa, descproducto, tipodetalle) { //
    addItem(idproducto, codproducto, 1, producto, opcionvendido, codmarca, marca, codmodelo, modelo, codpresentacion, presentacion, preciocompra, precioventa, descproducto, '+=');
}

/**
* Función que maneja la adición de productos al carrito desde los campos de entrada.
* Contiene la lógica de validación para entrada manual o desde autocompletado.
*/
function AgregaDetalle() {
    const code     = $('input#codproducto').val();
    const prod     = $('input#producto').val();
    let cantp      = parseFloat($('input#cantidad').val());
    const exist    = parseFloat($('input#existenciap').val());
    const prec     = $('input#preciocomprap').val();
    const prec2    = $('input#precioventap').val();
    const busqueda = $("#search_producto_combo").val();

    Utils.resetInputStyles(); // Resetear estilos antes de la validación

    if (code === "") {
        Utils.showAlert("POR FAVOR REALICE LA BÚSQUEDA DEL PRODUCTO CORRECTAMENTE!", "error", "#search_producto_combo");
        return false;
    } else if (isNaN(cantp) || cantp <= 0) {
        Utils.showAlert("POR FAVOR INGRESE UNA CANTIDAD VÁLIDA! (mayor a 0)", "error", "#cantidad");
        return false;
    } else if (cantp > exist) {
        Utils.showAlert("LA CANTIDAD DE PRODUCTOS SOLICITADO NO EXISTE EN ALMACEN, VERIFIQUE NUEVAMENTE POR FAVOR!", "error", "#cantidad");
        return false;
    } else {
        
        const Carrito = {
            Id:              $('input#idproducto').val(),
            Codigo:          $('input#codproducto').val(),
            Producto:        $('input#producto').val().replace(/[ '"]+/g, ' '),
            OpcionVendido:   $('input#opcionvendido').val(),
            Codmarca:        $('input#codmarca').val(),
            Marca:           $('input#marca').val(),
            Codmodelo:       $('input#codmodelo').val(),
            Modelo:          $('input#modelo').val(),
            Codpresentacion: $('input#codpresentacion').val(),
            Presentacion:    $('input#presentacion').val(),
            Precio:          $('input#preciocomprap').val(),
            Precio2:         $('input#precioventap').val(),
            Descproducto:    $('input#descproducto').val(),
            Cantidad:        $('input#cantidad').val(),
            opCantidad: '+='
        };
        _cartManager.updateCart(Carrito);
    }
}

/**
 * Maneja el cambio de cantidad en el input de una fila del carrito.
 * @param {HTMLInputElement} inputElement - El elemento input que disparó el evento.
 */
window.handleQuantityChange = function(inputElement) {
    const newQuantity = parseFloat(inputElement.value);

    // Si la cantidad es 0, o un número negativo, o no es un número, manejamos el error.
    // Opcionalmente, puedes permitir 0 para eliminar el ítem.
    if (isNaN(newQuantity) || newQuantity < 0) {
        // Revertir el valor del input a la última cantidad válida conocida o a 1
        inputElement.value = inputElement.defaultValue || 1; // Usar defaultValue o un valor por defecto
        // swal("Advertencia", "La cantidad debe ser un número válido mayor o igual a cero.", "warning");
        return;
    }

    // Extraer todas las propiedades del ítem de los atributos data-
    // Asegúrate de que los nombres de las propiedades coincidan con los que espera addItem
    const item = {
        id: inputElement.dataset.itemId,
        txtCodigo: inputElement.dataset.itemCodigo,
        producto: inputElement.dataset.itemProducto.replace(/&apos;/g, "'"), // Deshacer el escape de comillas simples
        opcionvendido: inputElement.dataset.itemOpcionVendido,
        codmarca: inputElement.dataset.itemCodmarca,
        marca: inputElement.dataset.itemMarca.replace(/&apos;/g, "'"),
        codmodelo: inputElement.dataset.itemCodmodelo,
        modelo: inputElement.dataset.itemModelo.replace(/&apos;/g, "'"),
        codpresentacion: inputElement.dataset.itemCodpresentacion,
        presentacion: inputElement.dataset.itemPresentacion.replace(/&apos;/g, "'"),
        precio: inputElement.dataset.itemPrecio,
        precio2: inputElement.dataset.itemPrecio2,
        descproducto: inputElement.dataset.itemDescproducto
    };

    // Actualizar el valor predeterminado del input para futuras comparaciones
    // (Útil si no se re-renderiza todo el carrito inmediatamente después del cambio)
    inputElement.defaultValue = newQuantity.toString();

    // Llamar a addItem para actualizar el carrito con la nueva cantidad absoluta
    addItem(
        item.id,
        item.txtCodigo,
        newQuantity.toString(), // Pasa la nueva cantidad como el argumento 'cantidad'
        item.producto,
        item.opcionvendido,
        item.codmarca,
        item.marca,
        item.codmodelo,
        item.modelo,
        item.codpresentacion,
        item.presentacion,
        item.precio,
        item.precio2,
        item.descproducto,
        '=' // Indica al backend que la operación es "establecer esta cantidad"
    );
};
// #endregion CORE FUNCTIONS

// #region EVENT HANDLERS
$(document).ready(function() {
    /*############ FUNCION DESACTIVA ENTER EN FORMULARIO ############*/
    $('#savecombos').keypress(function(e) { //
        const keycode = (e.keyCode ? e.keyCode : e.which);
        if (keycode === 13) {
            return false;
        }
    });

    /*############ FUNCION AGREGA POR BOTON ############*/
    $('#AgregaProducto').click(function() { //
        AgregaDetalle();
    });

    /*############ FUNCION AGREGA POR CRITERIO ############*/
    $('#cantidad, #search_producto_combo').keypress(function(e) { //
        const keycode = (e.keyCode ? e.keyCode : e.which);
        if (keycode === 13) {
            AgregaDetalle();
            e.preventDefault();
            return false;
        }
    });

    /* CANCELAR LOS ITEM AGREGADOS EN REGISTRO */
    $("#vaciar").click(function() { //
        _cartManager.clearCart();
    });

    if (!isEditingMode){// Si no es modo edición, entonces sí, inicializa el carrito vacío.
        // Cargar el carrito al iniciar la página
        _cartManager.updateCart({}); // Esto debería llamar a _uiUpdater.updateCartUI([], totals) y resetear.
    }


    // ###############################################################
    // # FUNCIONES DE TECLADO CON TECLAS DE FUNCIÓN (F-KEYS) #
    // ###############################################################
    $(document).keydown(function(e) {
        switch (e.keyCode) {
            case 13: // ENTER - Agrega detalle de producto
                // Modificado para evitar doble disparo si el foco está en un input o textarea
                if (e.target.id !== 'txtCodigoBarras' && e.target.tagName.toLowerCase() !== 'textarea' && e.target.tagName.toLowerCase() !== 'input') {
                    $('#AgregaProducto').trigger("click");
                    return false;
                }
            break;

            case 113: // F2 
                $('#btn-submit').trigger("click");
                return false;
            break;

            case 115: // F4 - Limpia valores de formulario (vaciar carrito)
                $('#vaciar').trigger("click");
                return false;
            break;
        }
    });
   
});
// #endregion EVENT HANDL